<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
/**
 * Autohive Theme Helper
 *
 *
 * @class        Autohive_Theme_Helper
 * @version      1.0
 * @category 	 Class
 * @author       ThemeTags
 */

if ( ! class_exists( 'Autohive_Theme_Helper' ) ) {
	class Autohive_Theme_Helper {

		/**
		 * Constructor
		 */
		function __construct() {
			add_action( 'autohive_after_body', array( $this, 'autohive_preloader_markup' ), 1 );
			add_action( 'wp_head', [ $this, 'track_post_views' ] );
			remove_action('wp_head', 'adjacent_posts_rel_link_wp_head', 10, 0);
		}


		/**
		 * Settings Compare function
		 *
		 * @param [type] $name
		 * @param boolean $check_key
		 * @param boolean $check_value
		 *
		 * @return void
		 */
		public static function options_compare( $name, $check_key = false, $check_value = false ) {
			$option = autohive_opt( $name );
			$meta   = get_post_meta( get_the_ID(), 'tt_page_options', true );

			//Check if check_key exist
			if ( $check_key ) {
				if ( $meta[ $check_key ] == $check_value ) {
					$option = $meta[ $name ];
				}

			} else {
				$var    = $meta[ $name ];
				$option = ! empty( $var ) ? $meta[ $name ] : autohive_opt( $name );
			}

			return $option;
		}

		/**
		 * Displays navigation to next/previous pages when applicable.
		 *
		 * @since autohive 1.0
		 */
		static function autohive_content_nav( $html_id ) {
			global $wp_query;
			$html_id = esc_attr( $html_id );
			if ( $wp_query->max_num_pages > 1 ) : ?>
				<nav id="<?php echo esc_attr( $html_id ); ?>" class="navigation" role="navigation">
					<h3 class="assistive-text"><?php esc_html_e( 'Post navigation', 'autohive' ); ?></h3>
					<div class="nav-previous"><?php next_posts_link( wp_kses( __( '<span class="meta-nav">&larr;</span> Older posts', 'autohive' ), array( 'span' => array( 'class' => array() ) ) ) ); ?></div>
					<div class="nav-next"><?php previous_posts_link( wp_kses( __( 'Newer posts <span class="meta-nav">&rarr;</span>', 'autohive' ), array( 'span' => array( 'class' => array() ) ) ) ); ?></div>
				</nav>
			<?php endif;
		}

		/* Pagination */

		static function autohive_post_pagination( $nav_query = false ) {

			global $wp_query, $wp_rewrite;

			// Don't print empty markup if there's only one page.
			if ( $GLOBALS['wp_query']->max_num_pages < 2 ) {
				return;
			}

			// Prepare variables
			$query        = $nav_query ? $nav_query : $wp_query;
			$max          = $query->max_num_pages;
			$current_page = max( 1, get_query_var( 'paged' ) );
			$big          = 999999;

			echo '<div class="template-pagination mt-50">';
			echo '' . paginate_links( array(
					'base'      => str_replace( $big, '%#%', esc_url( get_pagenum_link( $big ) ) ),
					'format'    => '?paged=%#%',
					'current'   => $current_page,
					'total'     => $max,
					'type'      => 'list',
					'prev_text' => '<i class="fa-solid fa-arrow-left"></i>',
					'next_text' => '<i class="fa-solid fa-arrow-right"></i>'
				) ) . ' ';
			echo '</div>';
		}


        public static function get_categories () {

	        $cats = get_terms( array(
		        'taxonomy' => 'category',
		        'hide_empty' => true
	        ));
	        $cat_array = [];

	        if ( is_array( $cats ) ) {
		        foreach ($cats as $cat) {
			        $cat_array[$cat->slug] = $cat->name;
		        }
	        }

	        return $cat_array;

        }



		/**
		 * Template for comments and pingbacks.
		 *
		 * To override this walker in a child theme without modifying the comments template
		 * simply create your own autohive_comment(), and that function will be used instead.
		 *
		 * Used as a callback by wp_list_comments() for displaying the comments.
		 *
		 * @since autohive 1.0
		 */
		static function autohive_comment( $comment, $args, $depth ) {
			$GLOBALS['comment'] = $comment;
			switch ( $comment->comment_type ) :
				case 'pingback' :
				case 'trackback' :
					// Display trackbacks differently than normal comments.
					?>
					<li <?php comment_class(); ?> id="comment-<?php comment_ID(); ?>">
					<p><?php esc_html_e( 'Pingback:', 'autohive' ); ?><?php comment_author_link(); ?><?php edit_comment_link( esc_html__( '(Edit)', 'autohive' ), '<span class="edit-link">', '</span>' ); ?></p>
					<?php
					break;
				default :
					// Proceed with normal comments.
					global $post;
					?>
					<li <?php comment_class(); ?> id="li-comment-<?php comment_ID(); ?>">
					<article id="comment-<?php comment_ID(); ?>" class="comment">
						<div class="comment-avatar">
							<?php echo get_avatar( $comment, 50 ); ?>
						</div>
						<div class="comment-info">
							<header class="comment-meta comment-author vcard">
								<?php
								printf( '<cite><b class="fn">%1$s</b> %2$s</cite>', get_comment_author_link(), // If current post author is also comment author, make it known visually.
								        ( $comment->user_id === $post->post_author ) ? '<span>' . esc_html__( 'Post author', 'autohive' ) . '</span>' : '' );
								printf( '<time datetime="%1$s">%2$s</time>', get_comment_time( 'c' ), /* translators: 1: date, 2: time */ sprintf( esc_html__( '%1$s at %2$s', 'autohive' ), get_comment_date(), get_comment_time() ) );
								?>
								<div class="reply">
									<?php comment_reply_link( array_merge( $args, array(
										'reply_text' => esc_html__( 'Reply', 'autohive' ),
										'after'      => '',
										'depth'      => $depth,
										'max_depth'  => $args['max_depth']
									) ) ); ?>
								</div><!-- .reply -->
							</header><!-- .comment-meta -->
							<?php if ( '0' == $comment->comment_approved ) : ?>
								<p class="comment-awaiting-moderation"><?php esc_html_e( 'Your comment is awaiting moderation.', 'autohive' ); ?></p>
							<?php endif; ?>
							<section class="comment-content comment">
								<?php comment_text(); ?>
								<?php edit_comment_link( esc_html__( 'Edit', 'autohive' ), '<p class="edit-link">', '</p>' ); ?>
							</section><!-- .comment-content -->
						</div>
					</article><!-- #comment-## -->
					<?php
					break;
			endswitch; // end comment_type check
		}

		/**
		 * Set up post entry meta.
		 *
		 * Prints HTML with meta information for current post: categories, tags, permalink, author, and date.
		 *
		 * Create your own autohive_entry_meta() to override in a child theme.
		 *
		 * @since autohive 1.0
		 */
		static function autohive_post_tags( $class ) {
			$tag_list       = get_the_tags();
			
			if( is_array( $tag_list ) && count( $tag_list ) > 0 ){
				echo '<div class="autohive_tags"><span class="me-2"><i class="fa-solid fa-tags"></i></span>';
				foreach( $tag_list as $tag ){
					echo '<a href="'. get_term_link( $tag->term_id ) .'" class="autohive_tag '. esc_attr( $class ) .'">'. esc_html( $tag->name ) .'</a>';
				}
				echo '</div>';
			}		

		}

		// Comment Number
		public static function autohive_get_comment_number(){
			$num_comments   = (int) get_comments_number();
			$write_comments = '';
			if ( comments_open() ) {
				if ( $num_comments == 0 ) {
					$comments = esc_html__( ' 0 Comment', 'autohive' );
				} elseif ( $num_comments > 1 ) {
					$comments = $num_comments . esc_html__( ' Comments', 'autohive' );
				} else {
					$comments = esc_html__( ' 1 Comment', 'autohive' );
				}
				$write_comments = '<a href="' . get_comments_link() . '" class="comment_count"><i class="fa-solid fa-comment"></i>' . $comments . '</a>';
			}
			return $write_comments;
		}

		static function autohive_entry_meta_small() {
			// Translators: used between list items, there is a space after the comma.
			$author          = sprintf( '<span class="author vcard"><a class="url fn n" href="%1$s" title="%2$s" rel="author">%3$s</a></span>', esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ), esc_attr( sprintf( wp_kses( __( 'View all posts by %s', 'autohive' ), array( 'a' => array() ) ), get_the_author() ) ), get_the_author() );
			$utility_text    = esc_html__( '%1$s', 'autohive' );
			printf( $utility_text, $author );
		}

		// Avatar ======================
		static function autohive_posted_author_avatar() {
			global $post;
			printf( '<a class="d-inline-flex align-items-center" href="%1$s"> %2$s <h6 class="ms-2 mb-0 autohive_post_author">%3$s</h6></a>', esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ), get_avatar( get_the_author_meta( 'ID' ), 35, '', '', array('class'=>'img-fluid rounded-circle') ), esc_html( get_the_author() ) );
		}

		public static function post_author_avatar( $size = 56, $default = '', $alt  = '', $args = '') {
			$post_author_id = get_post_field( 'post_author', get_the_ID() );
			echo get_avatar($post_author_id, $size, $default, $alt, $args);
		}


		// Post Author ================
		static function autohive_get_posted_by() {
			$autohive_post_author = sprintf( esc_html_x( '%s', 'post author', 'autohive' ), '<a class="autohive_post_author" href="' . esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ) . '">' . esc_html( get_the_author() ) . '</a>' );

			return $autohive_post_author;

		}

		// Post categories ==============
		static function autohive_entry_cat( $term = 'category', $class = '' ) {
			if ( 'post' === get_post_type() ) {
				$cats 	= get_the_terms(get_the_ID(), $term);		
				$class	= !empty( $class ) ? $class : 'cat_item';
				if( is_array( $cats ) ){
					foreach( $cats as $cat ){
						echo '<a href="'. esc_url( get_term_link( $cat->term_id, $term ) ) .'" class="'. esc_attr( $class ) .'">'. esc_html( $cat->name ) .'</a>';
					}
					
				}

			}

		}

		public static function get_the_post_tag( $class = 'tagcloud' ) {
			// Hide category and tag text for pages.
			if ( 'post' === get_post_type() ) {

				/* translators: used between list items, there is a space after the comma */
				$tags_list = get_the_tag_list( '', esc_html_x( ' ', 'list item separator', 'autohive' ) );
				if ( $tags_list ) {
					/* translators: 1: list of tags. */
					printf( '<div class="'.esc_attr( $class ).'"><span class="title text-secondary fw-bold me-2">' . esc_html( 'Tags:', 'autohive' ) . '</span>' . esc_html__( '%1$s', 'autohive' ) . '</div>', $tags_list ); // WPCS: XSS OK.
				}
			}
		}

		/**
		 * Trim text
		 */
		static function autohive_substring( $string, $limit, $afterlimit = '[...]' ) {
			if ( empty( $string ) ) {
				return $string;
			}
			$string = explode( ' ', strip_tags( $string ), $limit );

			if ( count( $string ) >= $limit ) {
				array_pop( $string );
				$string = implode( " ", $string ) . ' ' . $afterlimit;
			} else {
				$string = implode( " ", $string );
			}
			$string = preg_replace( '`[[^]]*]`', '', $string );

			return strip_shortcodes( $string );
		}

		/**
		 * Footer Widget Column.
		 *
		 * Set Column Footer Widget areas.
		 *
		 * @since autohive 1.0
		 */
		public function autohive_get_footer_column( $footer_columns ) {
			if ( $footer_columns == 12 ) {
				return 1;
			} elseif ( $footer_columns == 6 ) {
				return 2;
			} elseif ( $footer_columns == 4 ) {
				return 3;
			} else {
				return 4;
			}
		}

		/**
		 * Render sidebars.
		 *
		 * All Page Sidebar Control.
		 *
		 * @since autohive 1.0
		 */


		public static function render_sidebars( $args = 'blog' ) {
			$output        = array();
			$sidebar_style = '';

			$layout        = autohive_opt( $args . '_sidebar_layout' ) ? autohive_opt( $args . '_sidebar_layout' ) : 'right';
			$sidebar       = 'sidebar_widgets';
			$sidebar_width = autohive_opt( $args . '_sidebar_def_width', '8' );
			$sidebar_gap   = autohive_opt( $args . '_sidebar_gap' );
			$sidebar_class = $sidebar_style = '';

			if ( isset( $sidebar_gap ) && $sidebar_gap != 'def' && $layout != 'default' ) {
				$layout_pos    = $layout == 'left' ? 'right' : 'left';
				$sidebar_style = 'style="padding-' . $layout_pos . ': ' . $sidebar_gap . 'px;"';
			}

			$column = 12;
			if ( $layout == 'left' || $layout == 'right' ) {
				$column = (int) $sidebar_width;
			} else {
				$sidebar = '';
			}

			if ( ! is_active_sidebar( $sidebar ) ) {
				$column  = 12;
				$sidebar = '';
				$layout  = 'none';
			}

			$output['column']    = $column;
			$output['row_class'] = $layout != 'none' ? ' sidebar_' . esc_attr( $layout ) : '';
			$output['layout']    = $layout;
			$output['content']   = '';

			if ( $layout == 'left' || $layout == 'right' ) {
				$output['content'] .= '<div class="sidebar-container ' . $sidebar_class . 'col-lg-' . ( 12 - (int) $column ) . '" ' . $sidebar_style . '>';
				if ( is_active_sidebar( $sidebar ) ) {
					$output['content'] .= "<aside class='sidebar'>";
					ob_start();
					dynamic_sidebar( $sidebar );
					$output['content'] .= ob_get_clean();
					$output['content'] .= "</aside>";
				}
				$output['content'] .= "</div>";
			}

			return $output;
		}


		/**
		 * @param $name
		 * Name of dynamic sidebar
		 * Check sidebar is active then dynamic it.
		 */
		public static function generated_sidebar( $name ) {
			if ( is_active_sidebar( $name ) ) {
				dynamic_sidebar( $name );
			}
		}

		private function post_author( $author ) {
			if ( ! (bool) $author ) {
				return;
			}

			return '<span class="post_author">' . esc_html__( "by", "autohive" ) . ' <a href="' . esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ) . '">' . esc_html( get_the_author_meta( 'display_name' ) ) . '</a></span>';
		}

		/**
		 * Display Post Thumbnail.
		 */
		static function autohive_post_thumbnail() {
			if ( post_password_required() || is_attachment() || ! has_post_thumbnail() ) {
				return;
			}

			if ( is_singular() ) : ?>

				<div class="feature-image">
					<?php the_post_thumbnail( 'full' ); ?>
				</div><!-- .post-thumbnail -->

			<?php else : ?>
				<a class="post-thumbnail" href="<?php the_permalink(); ?>" aria-hidden="true" tabindex="-1">
					<?php the_post_thumbnail( 'full', array( 'alt' => the_title_attribute( array( 'echo' => false, ) ) ) ); ?>
				</a>
			<?php
			endif; // End is_singular().
		}


		static function render_author_info() {
			$user_email       = get_the_author_meta( 'user_email' );
			$user_avatar      = get_avatar( $user_email, 110 );
			$user_first       = get_the_author_meta( 'first_name' );
			$user_last        = get_the_author_meta( 'last_name' );
			$user_description = get_the_author_meta( 'description' );
			?>

			<div class="bd-author-info mt-60 rounded d-block d-sm-flex align-items-center">
				<?php echo get_avatar( get_the_author_meta( 'ID' ), 110, '', '', array('class'=>'img-fluid flex-shrink-0 rounded-circle') ); ?>
				<div class="bd-author-info-right mt-3 mt-sm-0 ms-sm-3">
					<h6><?php echo get_the_author(); ?></h6>
					<p><?php echo esc_html( $user_description ); ?></p>					
				</div>
			</div>

			<?php
			

		}

		/**
		 * Preloader.
		 *
		 * Preloader Init.
		 *
		 * @since autohive 1.0
		 */

		public function autohive_preloader_markup() {
			$preloader      = autohive_opt( 'preloader' );
			$preloader_opt  = autohive_opt( 'autohive_preloader' );
			$preloader_type = autohive_opt( 'preloader-type' );
			$preloader_img  = autohive_opt( 'preloader-images' );


			if ( ! empty( $preloader_opt ) ) :
				$style_name = substr( $preloader_opt, 0, - 2 );
				$style_div  = substr( $preloader_opt, - 1 );
				if ( $preloader ) : ?>

					<div id="preloader">
						<?php if ( $preloader_type == 'css' ) : ?>
							<div id="loader">
								<div class="loader-inner <?php echo esc_attr( $style_name ); ?>">
									<?php for ( $div = 0; $div < $style_div; $div ++ ) : ?>
										<div></div>
									<?php endfor; ?>
								</div>
							</div><!-- /#loader -->
						<?php elseif ( $preloader_type == 'preloader-img' ) : ?>
							<?php $img = wp_get_attachment_image_src( $preloader_img, 'full', true ); ?>

							<img class="pr" src="<?php echo esc_url( $img[0] ); ?>"
									width="<?php echo esc_attr( $img[1] ); ?>"
									height="<?php echo esc_attr( $img[2] ); ?>" alt="<?php get_bloginfo( 'name' ); ?>"/>
						<?php endif; ?>
					</div><!-- /#preloader -->
				<?php
				endif;
			endif;
		}


		public static function render_html( $args ) {
			return isset( $args ) ? $args : '';
		}

		static function autohive_entry_comments( $post_id, $no_comments = 'No Comments' ) {

			$comments_number = get_comments_number( $post_id );
			if ( $comments_number == 0 ) {
				$comment_text = $no_comments;
			} elseif ( $comments_number == 1 ) {
				$comment_text = esc_html__( '1 Comment', 'autohive' );
			} elseif ( $comments_number > 1 ) {
				$comment_text = $comments_number . esc_html__( ' Comments', 'autohive' );
			}
			echo '<a href="'. esc_url( get_comments_link( $post_id ) ) .'">'. esc_html( $comment_text ) .'</a>';

		}

		/** Post Archive Link */
		static function autohive_archive_url(){

			$archive_year  = get_the_time( 'Y' ); 
			$archive_month = get_the_time( 'm' ); 
			$archive_day   = get_the_time( 'd' );
			
			return '<a href="'. esc_url( get_day_link( $archive_year, $archive_month, $archive_day ) ) .'">'. get_the_date() .'</a>';
		}


		/**
		 * Logo
		 */
		public static function branding_logo( $class = '' ) {
			
			$logo_main        = autohive_opt( 'main_logo' );
			$sticky           = autohive_opt( 'sticky_logo' );
			$retina_logo      = autohive_opt( 'retina_logo' );
			$retina_sticky    = autohive_opt( 'retina_logo_sticky' );
			$is_sticky_header = autohive_opt( 'header_sticky' );

			// Logo Callback
			$logo               = ! empty( $logo_main['url'] ) ? $logo_main['url'] : '';
			$logo_contrast      = ! empty( $sticky['url'] ) ? $sticky['url'] : '';
			$retina_logo        = ! empty( $retina_logo['url'] ) ? $retina_logo['url'] : '';
			$retina_logo_sticky = ! empty( $retina_sticky ['url'] ) ? $retina_sticky ['url'] : '';

			// Logo Meta Callback
			$meta = get_post_meta( get_the_ID(), 'tt_page_options', true );

			$page_logo = isset( $meta['page_logo'] ) ? $meta['page_logo'] : '0';

			if ( $page_logo ) {
				$logo               = ! empty( $meta['meta_main_logo']['url'] ) ? $meta['meta_main_logo']['url'] : $logo;
				$logo_contrast      = ! empty( $meta['meta_sticky_logo']['url'] ) ? $meta['meta_sticky_logo']['url'] : $logo_contrast;
				$retina_logo        = ! empty( $meta['retina_logo']['url'] ) ? $meta['retina_logo']['url'] : $retina_logo;
				$retina_logo_sticky = ! empty( $meta['retina_logo_sticky']['url'] ) ? $meta['retina_logo_sticky']['url'] : $retina_logo_sticky;
			} 
			$class .= !empty( $logo_contrast ) ? ' duble_logo' : ' only_main_logo';
			$a_class = !empty( $class ) ? $class : '';
			?>

			<a href="<?php echo esc_url( home_url( '/' ) ); ?>" class="<?php echo esc_attr( $a_class ); ?>" rel="home">
				<?php 
				if ( ! empty( $logo ) ) { ?>
					<img <?php if ( ! empty( $retina_logo ) ) {
						echo 'srcset="' . esc_attr( $retina_logo ) . ' 2x"';
					} ?> src="<?php echo esc_url( $logo ); ?>" alt="<?php esc_attr_e( get_bloginfo( 'name' ) ); ?>"
							class="main-logo"/>

					<?php if ( $logo_contrast !== '' && $is_sticky_header == true ): ?>
						<img <?php if ( ! empty( $retina_logo_sticky ) ) {
							echo 'srcset="' . esc_attr( $retina_logo_sticky ) . ' 2x"';
						} ?> src="<?php echo esc_url( $logo_contrast ); ?>"
								alt="<?php esc_attr_e( get_bloginfo( 'name' ) ); ?>" class="logo-sticky">
					<?php endif; ?>

				<?php } else { ?>
					<h3><?php bloginfo( 'name' ); ?></h3>
				<?php } ?>
			</a>
			<?php
		}

		/**
		 * @return void
         *
         * Social share icons
         *
         * @access public
		 */
		public static function get_the_social_share_links() {
            ?>
            <div class="bs_social_share">
                <a href="https://facebook.com/sharer/sharer.php?u=<?php the_permalink(); ?>"><i class="fab fa-facebook-f"></i></a>
                <a href="http://www.linkedin.com/shareArticle?mini=true&url=<?php the_permalink(); ?>"><i class="fab fa-linkedin-in"></i></a>
                <a href="https://twitter.com/intent/tweet?text=<?php the_permalink(); ?>"><i class="fab fa-twitter"></i></a>
                <a href="https://www.pinterest.com/pin/create/button/?url=<?php the_permalink() ?>"><i class="fab fa-twitter"></i></a>
            </div>
			<?php
		}

		/**
		 * Get the first category name
		 * @param string $term
		 */
		public static function the_first_category($term = 'category') {
			$cats = get_the_terms(get_the_ID(), $term);
			$cat  = is_array($cats) ? $cats[0]->name : '';
			echo esc_html($cat);
		}

		/**
		 * Get the first category link
		 * @param string $term
		 */
		public static function the_first_category_link($term='category') {
			$cats = get_the_terms(get_the_ID(), $term);
			$cat  = is_array($cats) ? get_category_link($cats[0]->term_id) : '';
			echo esc_url($cat);
		}

		/**
		 * Returns array of title tags
		 *
		 * @param bool $first_empty
		 * @param array $additional_elements
		 *
		 * @return array
		 */
		static function tt_get_title_tag( $first_empty = false, $additional_elements = array() ) {
			$title_tag = array();

			if ( $first_empty ) {
				$title_tag[''] = esc_html__( 'Default', 'autohive' );
			}

			$title_tag['h1'] = 'h1';
			$title_tag['h2'] = 'h2';
			$title_tag['h3'] = 'h3';
			$title_tag['h4'] = 'h4';
			$title_tag['h5'] = 'h5';
			$title_tag['h6'] = 'h6';

			if ( ! empty( $additional_elements ) ) {
				$title_tag = array_merge( $title_tag, $additional_elements );
			}

			return $title_tag;
		}

		public static function get_post_list( $post_type = 'post' ) {
			$post_list = ['' => esc_html__( 'Default', 'autohive' ) ];
			$posts = get_posts(
				[
					'posts_per_page' => - 1,
					'post_type'      => $post_type,
					'orderby'        => 'name',
					'order'          => 'ASC'
				]
			);
			foreach ( $posts as  $value ) {
				$post_list[$value->ID] = $value->post_title;
			}
			return $post_list;
		}

		// Listing Body Type
		public static function listing_body_type() {
			$options = get_option('autohive_cs_opt');
			$bodyType = $options['car_body_type_group'];
			
			if ( is_array($bodyType) && !empty( $bodyType ) ) {
				$btype = array();
				foreach ($bodyType as $item) {
					$type_slug = sanitize_title_with_dashes($item['body_type_name']);
					$btype[$type_slug] = $item['body_type_name'];
				}
				return $btype;
			}
		}
		
		public static function car_pickup_location() {
			$options = get_option('autohive_cs_opt');
			$bodyType = $options['car_pickup_locations'];
			
			if ( is_array($bodyType) && !empty( $bodyType ) ) {
				$btype = array();
				foreach ($bodyType as $item) {
					$type_slug = sanitize_title_with_dashes($item['location']);
					$btype[$type_slug] = $item['location'];
				}
				return $btype;
			}
		}


		public static function autohive_render_footer($footer_style){
			$elementor_instance = Elementor\Plugin::instance();
			return $elementor_instance->frontend->get_builder_content_for_display( $footer_style );
		}

		/* Get Nav menus */
		public static function autohive_get_nav_menus(){
			$nav_menus = wp_get_nav_menus();
			if( is_array( $nav_menus ) ){
				$menu_location[] = '';
				foreach( $nav_menus as $item ){
					$menu_location[$item->slug] = $item->name;
				}
				return $menu_location;
			}
		}


		// Get comment count text
		public static function get_the_comment_count( $post_id ) {
			$comments_number = get_comments_number($post_id);
			if ( $comments_number == 0) {
				$comment_text = esc_html__( 'No Comment', 'rave' );
			} elseif( $comments_number == 1) {
				$comment_text = esc_html__( 'Comment (1)', 'rave' );
			} elseif( $comments_number > 1) {
				$comment_text = esc_html__( 'Comments ('.$comments_number.')', 'rave' );
			}

            return $comment_text;
		}


        public static function get_the_post_type_lebel_breadcrumb() {

	        $post_type_obj_page = get_post_type_object('page' );

	        $post_type_obj_post = get_post_type_object('post' );

            if ( !empty($post_type_obj_page) ) {
	            $label_name = $post_type_obj_page->label;
            }
            elseif ( $post_type_obj_post ) {
                $label_name = $post_type_obj_post->label;
            }

            return $label_name;

        }


		/**
		 * Post View Count ==================================
		 */
		public static function set_post_views( $post_id )
		{
			$count_key = 'autohive_post_views_count';
			$count = get_post_meta( $post_id, $count_key, true );
		
			if ( $count == '' ) {
				$count = 0;
				delete_post_meta( $post_id, $count_key );
				add_post_meta( $post_id, $count_key, '0' );
			} else {
				$count++;
				update_post_meta( $post_id, $count_key, $count );
			}
		}

		public static function track_post_views($post_id)
		{
			if ( !is_single() ) return;
			if ( empty($post_id) ) {
				global $post;
				$post_id = $post->ID;
			}
			self::set_post_views($post_id);
		}

		public static function get_post_views( $post_id )
		{
			$count_key = 'autohive_post_views_count';
			$count = get_post_meta( $post_id, $count_key, true);
			if ($count == '') {
				delete_post_meta( $post_id, $count_key);
				add_post_meta( $post_id, $count_key, '0');
				return "0 View";
			}
			return $count . ' Views';
		}


	}

	// Instantiate theme
	new Autohive_Theme_Helper();
}