<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
/**
 * Autohive Theme Helper
 *
 *
 * @class        Autohive_Theme_Helper
 * @version      1.0
 * @category 	 Class
 * @author       ThemeTags
 */

if ( ! class_exists( 'Autohive_Theme_Helper' ) ) {
	class Autohive_Theme_Helper {

		/**
		 * Constructor
		 */
		function __construct() {
			add_action( 'wp_head', [ $this, 'track_post_views' ] );
			remove_action('wp_head', 'adjacent_posts_rel_link_wp_head', 10, 0);
		}


		/* Pagination */
		static function autohive_post_pagination( $nav_query = false ) {

			global $wp_query, $wp_rewrite;

			// Don't print empty markup if there's only one page.
			if ( $GLOBALS['wp_query']->max_num_pages < 2 ) {
				return;
			}

			// Prepare variables
			$query        = $nav_query ? $nav_query : $wp_query;
			$max          = $query->max_num_pages;
			$current_page = max( 1, get_query_var( 'paged' ) );
			$big          = 999999;

			echo '<div class="template-pagination mt-50">';
			echo '' . paginate_links( array(
					'base'      => str_replace( $big, '%#%', esc_url( get_pagenum_link( $big ) ) ),
					'format'    => '?paged=%#%',
					'current'   => $current_page,
					'total'     => $max,
					'type'      => 'list',
					'prev_text' => '<i class="fa-solid fa-arrow-left"></i>',
					'next_text' => '<i class="fa-solid fa-arrow-right"></i>'
				) ) . ' ';
			echo '</div>';
		}


        public static function get_categories () {

	        $cats = get_terms( array(
		        'taxonomy' => 'category',
		        'hide_empty' => true
	        ));
	        $cat_array = [];

	        if ( is_array( $cats ) ) {
		        foreach ($cats as $cat) {
			        $cat_array[$cat->slug] = $cat->name;
		        }
	        }

	        return $cat_array;

        }



		/**
		 * Template for comments and pingbacks.
		 *
		 * To override this walker in a child theme without modifying the comments template
		 * simply create your own autohive_comment(), and that function will be used instead.
		 *
		 * Used as a callback by wp_list_comments() for displaying the comments.
		 *
		 * @since autohive 1.0
		 */
		static function autohive_comment( $comment, $args, $depth ) {
			$GLOBALS['comment'] = $comment;
			switch ( $comment->comment_type ) :
				case 'pingback' :
				case 'trackback' :
					// Display trackbacks differently than normal comments.
					?>
					<li <?php comment_class(); ?> id="comment-<?php comment_ID(); ?>">
					<p><?php esc_html_e( 'Pingback:', 'autohive' ); ?><?php comment_author_link(); ?><?php edit_comment_link( esc_html__( '(Edit)', 'autohive' ), '<span class="edit-link">', '</span>' ); ?></p>
					<?php
					break;
				default :
					// Proceed with normal comments.
					global $post;
					?>
					<li <?php comment_class(); ?> id="li-comment-<?php comment_ID(); ?>">
					<article id="comment-<?php comment_ID(); ?>" class="comment">
						<div class="comment-avatar">
							<?php echo get_avatar( $comment, 50 ); ?>
						</div>
						<div class="comment-info">
							<header class="comment-meta comment-author vcard">
								<?php
								printf( '<cite><b class="fn">%1$s</b> %2$s</cite>', get_comment_author_link(), // If current post author is also comment author, make it known visually.
								        ( $comment->user_id === $post->post_author ) ? '<span>' . esc_html__( 'Post author', 'autohive' ) . '</span>' : '' );
								printf( '<time datetime="%1$s">%2$s</time>', get_comment_time( 'c' ), /* translators: 1: date, 2: time */ sprintf( esc_html__( '%1$s at %2$s', 'autohive' ), get_comment_date(), get_comment_time() ) );
								?>
								<div class="reply">
									<?php comment_reply_link( array_merge( $args, array(
										'reply_text' => esc_html__( 'Reply', 'autohive' ),
										'after'      => '',
										'depth'      => $depth,
										'max_depth'  => $args['max_depth']
									) ) ); ?>
								</div><!-- .reply -->
							</header><!-- .comment-meta -->
							<?php if ( '0' == $comment->comment_approved ) : ?>
								<p class="comment-awaiting-moderation"><?php esc_html_e( 'Your comment is awaiting moderation.', 'autohive' ); ?></p>
							<?php endif; ?>
							<section class="comment-content comment">
								<?php comment_text(); ?>
								<?php edit_comment_link( esc_html__( 'Edit', 'autohive' ), '<p class="edit-link">', '</p>' ); ?>
							</section><!-- .comment-content -->
						</div>
					</article><!-- #comment-## -->
					<?php
					break;
			endswitch; // end comment_type check
		}

		public static function post_author_avatar( $size = 56, $default = '', $alt  = '', $args = '') {
			$post_author_id = get_post_field( 'post_author', get_the_ID() );
			echo get_avatar($post_author_id, $size, $default, $alt, $args);
		}


		public static function get_the_post_tag( $class = 'tagcloud' ) {
			// Hide category and tag text for pages.
			if ( 'post' === get_post_type() ) {

				/* translators: used between list items, there is a space after the comma */
				$tags_list = get_the_tag_list( '', esc_html_x( ' ', 'list item separator', 'autohive' ) );
				if ( $tags_list ) {
					/* translators: 1: list of tags. */
					printf( '<div class="'.esc_attr( $class ).'"><span class="title text-secondary fw-bold me-2">' . esc_html( 'Tags:', 'autohive' ) . '</span>' . esc_html__( '%1$s', 'autohive' ) . '</div>', $tags_list ); // WPCS: XSS OK.
				}
			}
		}

		/**
		 * Trim text
		 */
		static function autohive_substring( $string, $limit, $afterlimit = '[...]' ) {
			if ( empty( $string ) ) {
				return $string;
			}
			$string = explode( ' ', strip_tags( $string ), $limit );

			if ( count( $string ) >= $limit ) {
				array_pop( $string );
				$string = implode( " ", $string ) . ' ' . $afterlimit;
			} else {
				$string = implode( " ", $string );
			}
			$string = preg_replace( '`[[^]]*]`', '', $string );

			return strip_shortcodes( $string );
		}

		/**
		 * Render sidebars.
		 *
		 * All Page Sidebar Control.
		 *
		 * @since autohive 1.0
		 */


		public static function render_sidebars( $args = 'blog' ) {
			$output        = array();
			$sidebar_style = '';

			$layout        = autohive_opt( $args . '_sidebar_layout' ) ? autohive_opt( $args . '_sidebar_layout' ) : 'right';
			$sidebar       = 'sidebar_widgets';
			$sidebar_width = autohive_opt( $args . '_sidebar_def_width', '8' );
			$sidebar_gap   = autohive_opt( $args . '_sidebar_gap' );
			$sidebar_class = $sidebar_style = '';

			if ( isset( $sidebar_gap ) && $sidebar_gap != 'def' && $layout != 'default' ) {
				$layout_pos    = $layout == 'left' ? 'right' : 'left';
				$sidebar_style = 'style="padding-' . $layout_pos . ': ' . $sidebar_gap . 'px;"';
			}

			$column = 12;
			if ( $layout == 'left' || $layout == 'right' ) {
				$column = (int) $sidebar_width;
			} else {
				$sidebar = '';
			}

			if ( ! is_active_sidebar( $sidebar ) ) {
				$column  = 12;
				$sidebar = '';
				$layout  = 'none';
			}

			$output['column']    = $column;
			$output['row_class'] = $layout != 'none' ? ' sidebar_' . esc_attr( $layout ) : '';
			$output['layout']    = $layout;
			$output['content']   = '';

			if ( $layout == 'left' || $layout == 'right' ) {
				$output['content'] .= '<div class="sidebar-container ' . $sidebar_class . 'col-lg-' . ( 12 - (int) $column ) . ' mt-5 mt-xl-0" ' . $sidebar_style . '>';
				if ( is_active_sidebar( $sidebar ) ) {
					$output['content'] .= "<aside class='sidebar'>";
					ob_start();
					dynamic_sidebar( $sidebar );
					$output['content'] .= ob_get_clean();
					$output['content'] .= "</aside>";
				}
				$output['content'] .= "</div>";
			}

			return $output;
		}


		/**
		 * @param $name
		 * Name of dynamic sidebar
		 * Check sidebar is active then dynamic it.
		 */
		public static function generated_sidebar( $name ) {
			if ( is_active_sidebar( $name ) ) {
				dynamic_sidebar( $name );
			}
		}

		private function post_author( $author ) {
			if ( ! (bool) $author ) {
				return;
			}

			return '<span class="post_author">' . esc_html__( 'by', 'autohive' ) . ' <a href="' . esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ) . '">' . esc_html( get_the_author_meta( 'display_name' ) ) . '</a></span>';
		}

		/**
		 * Display Post Thumbnail.
		 */
		static function autohive_post_thumbnail() {
			if ( post_password_required() || is_attachment() || ! has_post_thumbnail() ) {
				return;
			}

			if ( is_singular() ) : ?>

				<div class="feature-image">
					<?php the_post_thumbnail( 'full' ); ?>
				</div><!-- .post-thumbnail -->

			<?php else : ?>
				<a class="post-thumbnail" href="<?php the_permalink(); ?>" aria-hidden="true" tabindex="-1">
					<?php the_post_thumbnail( 'full', array( 'alt' => the_title_attribute( array( 'echo' => false, ) ) ) ); ?>
				</a>
			<?php
			endif; // End is_singular().
		}

		public static function render_html( $args ) {
			return isset( $args ) ? $args : '';
		}

		/** Post Archive Link */
		static function autohive_archive_url(){
			$archive_year  = get_the_time( 'Y' ); 
			$archive_month = get_the_time( 'm' ); 
			$archive_day   = get_the_time( 'd' );
			
			return '<a href="'. esc_url( get_day_link( $archive_year, $archive_month, $archive_day ) ) .'">'. get_the_date() .'</a>';
		}


		/**
		 * Logo
		 */
		public static function branding_logo( $class = '' ) {
			
			$logo_main        = autohive_opt( 'main_logo' );
			$sticky           = autohive_opt( 'sticky_logo' );
			$retina_logo      = autohive_opt( 'retina_logo' );
			$retina_sticky    = autohive_opt( 'retina_logo_sticky' );
			$is_sticky_header = autohive_opt( 'header_sticky' );

			// Logo Callback
			$logo               = ! empty( $logo_main['url'] ) ? $logo_main['url'] : '';
			$logo_contrast      = ! empty( $sticky['url'] ) ? $sticky['url'] : '';
			$retina_logo        = ! empty( $retina_logo['url'] ) ? $retina_logo['url'] : '';
			$retina_logo_sticky = ! empty( $retina_sticky ['url'] ) ? $retina_sticky ['url'] : '';

			// Logo Meta Callback
			$meta = get_post_meta( get_the_ID(), 'tt_page_options', true );
			$page_logo = isset( $meta['page_logo'] ) ? $meta['page_logo'] : '';

			if ( $page_logo ) {
				$logo               = ! empty( $meta['meta_main_logo']['url'] ) ? $meta['meta_main_logo']['url'] : $logo;
				$logo_contrast      = ! empty( $meta['meta_sticky_logo']['url'] ) ? $meta['meta_sticky_logo']['url'] : $logo_contrast;
				$retina_logo        = ! empty( $meta['retina_logo']['url'] ) ? $meta['retina_logo']['url'] : $retina_logo;
				$retina_logo_sticky = ! empty( $meta['retina_logo_sticky']['url'] ) ? $meta['retina_logo_sticky']['url'] : $retina_logo_sticky;
			}

			$class .= !empty( $logo_contrast ) ? ' duble_logo' : ' only_main_logo';
			$a_class = !empty( $class ) ? $class : '';

            $retina_logo_srcset = !empty($retina_logo) ? 'srcset='.esc_attr($retina_logo) : '';
            $retina_logo_sticky_srcset = !empty($retina_logo_sticky) ? 'srcset='.esc_attr($retina_logo_sticky) : '';
			?>
			<a href="<?php echo esc_url( home_url( '/' ) ); ?>" class="<?php echo esc_attr( $a_class ); ?>" rel="home">
				<?php 
				if ( ! empty( $logo ) ) { ?>
                    <img class="main-logo" src="<?php echo esc_url( $logo ); ?>" <?php echo autohive_kses_post($retina_logo_srcset) ?> alt="<?php bloginfo('name') ?>">

                    <?php if ( $logo_contrast !== '' && $is_sticky_header == true ) : ?>
						<img class="logo-sticky" src="<?php echo esc_url( $logo_contrast ); ?>" <?php echo autohive_kses_post($retina_logo_sticky_srcset) ?> alt="<?php bloginfo('name'); ?>">
					<?php endif; ?>
				    <?php
                } else { ?>
					<h3 class="text-white"><?php bloginfo( 'name' ); ?></h3>
				    <?php
                }
                ?>
			</a>
			<?php
		}

		/**
		 * @return void
         *
         * Social share icons
         *
         * @access public
		 */
		public static function get_the_social_share_links() {
            ?>
            <div class="bs_social_share">
                <a href="https://facebook.com/sharer/sharer.php?u=<?php the_permalink(); ?>"><i class="fab fa-facebook-f"></i></a>
                <a href="http://www.linkedin.com/shareArticle?mini=true&url=<?php the_permalink(); ?>"><i class="fab fa-linkedin-in"></i></a>
                <a href="https://twitter.com/intent/tweet?text=<?php the_permalink(); ?>"><i class="fab fa-twitter"></i></a>
                <a href="https://www.pinterest.com/pin/create/button/?url=<?php the_permalink() ?>"><i class="fab fa-twitter"></i></a>
            </div>
			<?php
		}

		/**
		 * Get the first category name
		 * @param string $term
		 */
		public static function the_first_category($term = 'category') {
			$cats = get_the_terms(get_the_ID(), $term);
			$cat  = is_array($cats) ? $cats[0]->name : '';
			echo esc_html($cat);
		}

		/**
		 * Get the first category link
		 * @param string $term
		 */
		public static function the_first_category_link($term='category') {
			$cats = get_the_terms(get_the_ID(), $term);
			$cat  = is_array($cats) ? get_category_link($cats[0]->term_id) : '';
			echo esc_url($cat);
		}

		/**
		 * Returns array of title tags
		 *
		 * @param bool $first_empty
		 * @param array $additional_elements
		 *
		 * @return array
		 */
		static function tt_get_title_tag( $first_empty = false, $additional_elements = array() ) {
			$title_tag = array();

			if ( $first_empty ) {
				$title_tag[''] = esc_html__( 'Default', 'autohive' );
			}

			$title_tag['h1'] = 'h1';
			$title_tag['h2'] = 'h2';
			$title_tag['h3'] = 'h3';
			$title_tag['h4'] = 'h4';
			$title_tag['h5'] = 'h5';
			$title_tag['h6'] = 'h6';

			if ( ! empty( $additional_elements ) ) {
				$title_tag = array_merge( $title_tag, $additional_elements );
			}

			return $title_tag;
		}

		public static function get_post_list( $post_type = 'post' ) {
			$post_list = ['' => esc_html__( 'Default', 'autohive' ) ];
			$posts = get_posts(
				[
					'posts_per_page' => - 1,
					'post_type'      => $post_type,
					'orderby'        => 'name',
					'order'          => 'ASC'
				]
			);
			foreach ( $posts as  $value ) {
				$post_list[$value->ID] = $value->post_title;
			}
			return $post_list;
		}

		// Listing Body Type
		public static function listing_body_type() {
			$options = get_option('autohive_cs_opt');
			
			if( !empty( $options ) && isset($options['car_body_type_group']) ){
				$bodyType = $options['car_body_type_group'];
				
				if ( is_array($bodyType) && !empty( $bodyType ) ) {
					$btype = array();
					foreach ($bodyType as $item) {
						$type_slug = sanitize_title_with_dashes($item['body_type_name']);
						$btype[$type_slug] = $item['body_type_name'];
					}
					return $btype;
				}
			}
		}
		
		public static function car_pickup_location() {
			$options = get_option('autohive_cs_opt');
			if (!empty($options) && isset($options['car_pickup_locations']) ) {
				$bodyType = $options['car_pickup_locations'];
				
				if ( is_array($bodyType) && !empty( $bodyType ) ) {
					$btype = array();
					foreach ($bodyType as $item) {
						$type_slug = sanitize_title_with_dashes($item['location']);
						$btype[$type_slug] = $item['location'];
					}
					return $btype;
				}
			}
		}


		public static function autohive_render_footer($footer_style){
			$elementor_instance = Elementor\Plugin::instance();
			return $elementor_instance->frontend->get_builder_content_for_display( $footer_style );
		}

		/* Get Nav menus */
		public static function autohive_get_nav_menus(){
			$nav_menus = wp_get_nav_menus();
			if( is_array( $nav_menus ) ){
				$menu_location[] = '';
				foreach( $nav_menus as $item ){
					$menu_location[$item->slug] = $item->name;
				}
				return $menu_location;
			}
		}


		// Get comment count text
		public static function get_the_comment_count( $post_id ) {
			$comments_number = get_comments_number($post_id);
			if ( $comments_number == 0) {
				$comment_text = __( 'No Comment', 'autohive' );
			} elseif( $comments_number == 1) {
				$comment_text = __( 'Comment (1)', 'autohive' );
			} elseif( $comments_number > 1) {
				$comment_text = __( 'Comments ', 'autohive' ) . "($comments_number)";
			}

            return $comment_text;
		}


		/**
		 * Post View Count ==================================
		 */
		public static function set_post_views( $post_id )
		{
			$count_key = 'autohive_post_views_count';
			$count = get_post_meta( $post_id, $count_key, true );
		
			if ( $count == '' ) {
				$count = 0;
				delete_post_meta( $post_id, $count_key );
				add_post_meta( $post_id, $count_key, '0' );
			} else {
				$count++;
				update_post_meta( $post_id, $count_key, $count );
			}
		}

		public static function track_post_views($post_id)
		{
			if ( !is_single() ) return;
			if ( empty($post_id) ) {
				global $post;
				$post_id = $post->ID;
			}
			self::set_post_views($post_id);
		}

		public static function get_post_views( $post_id )
		{
			$count_key = 'autohive_post_views_count';
			$count = get_post_meta( $post_id, $count_key, true);
			if ($count == '') {
				delete_post_meta( $post_id, $count_key);
				add_post_meta( $post_id, $count_key, '0');
				return "0 View";
			}
			return $count . ' Views';
		}


		// Post categories ==============
		public static function autohive_entry_cat( $term = 'category', $class = '' ) {
			if ( 'post' === get_post_type() ) {
				$cats 	= get_the_terms(get_the_ID(), $term);
				$class	= !empty( $class ) ? $class : 'cat_item';
				if( is_array( $cats ) ){
					foreach( $cats as $cat ){
						echo '<a href="'. esc_url( get_term_link( $cat->term_id, $term ) ) .'" class="'. esc_attr( $class ) .'">'. esc_html( $cat->name ) .'</a>';
					}
				}

			}

		}


	}

	// Instantiate theme
	new Autohive_Theme_Helper();
}