<?php
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}


/**
 * Class Autohive_Core_Helper.
 */

if ( ! class_exists( 'Autohive_Core_Helper' ) ) {
	class Autohive_Core_Helper {

		protected static $instance  = null;
		protected static $post_type = 'job';
		protected static $tag       = 'portfolio_tags';
		protected static $category  = 'portfolio_category';

		public static function instance() {
			if ( null === self::$instance ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

		

		public function __construct()
		{
			add_filter( 'upload_mimes', [ $this, 'autohive_svg_mime_types'] );
			

		}

		protected function build_extra_terms_query( $query_args, $taxonomies ) {
			if ( empty( $taxonomies ) ) {
				return $query_args;
			}

			$terms       = explode( ', ', $taxonomies );
			$tax_queries = array(); // List of taxonomies.

			if ( ! isset( $query_args['tax_query'] ) ) {
				$query_args['tax_query'] = array();

				foreach ( $terms as $term ) {
					$tmp       = explode( ':', $term );
					$taxonomy  = $tmp[0];
					$term_slug = $tmp[1];
					if ( ! isset( $tax_queries[ $taxonomy ] ) ) {
						$tax_queries[ $taxonomy ] = array(
							'taxonomy' => $taxonomy,
							'field'    => 'slug',
							'terms'    => array( $term_slug ),
						);
					} else {
						$tax_queries[ $taxonomy ]['terms'][] = $term_slug;
					}
				}
				$query_args['tax_query'] = array_values( $tax_queries );
				$query_args['tax_query']['relation'] = 'AND';
			} else {
				foreach ( $terms as $term ) {
					$tmp       = explode( ':', $term );
					$taxonomy  = $tmp[0];
					$term_slug = $tmp[1];

					foreach ( $query_args['tax_query'] as $key => $query ) {
						if ( is_array( $query ) ) {
							if ( $query['taxonomy'] == $taxonomy ) {
								$query_args['tax_query'][ $key ]['terms'][] = $term_slug;
								break;
							} else {
								$query_args['tax_query'][] = array(
									'taxonomy' => $taxonomy,
									'field'    => 'slug',
									'terms'    => array( $term_slug ),
								);
							}
						}
					}
				}
			}

			return $query_args;
		}

		/**
		 * @param $taxonomy
		 * @param $helper
		 *
		 * @since 1.0
		 */
		public static function get_term_parents_list($term_id, $taxonomy, $args = []) {
			$list = '';
			$term = get_term( $term_id, $taxonomy );

			if (is_wp_error($term)) {
				return $term;
			}

			if (! $term) {
				return $list;
			}

			$term_id = $term->term_id;

			$defaults = [
				'format' => 'name',
				'separator' => '/',
				'inclusive' => true,
			];

			$args = wp_parse_args( $args, $defaults );

			foreach (['inclusive'] as $bool) {
				$args[ $bool ] = wp_validate_boolean( $args[ $bool ] );
			}

			$parents = get_ancestors( $term_id, $taxonomy, 'taxonomy' );

			if ($args['inclusive']) {
				array_unshift( $parents, $term_id );
			}

			$a = count($parents) - 1;
			foreach (array_reverse( $parents ) as $index => $term_id) {
				$parent = get_term( $term_id, $taxonomy );
				$temp_sep = $args['separator'];
				$lastElement = reset($parents);
				$first = end($parents);

				if ($index == $a - 1) {
					$temp_sep = '';
				}

				if ($term_id != $lastElement) {
					$name = $parent->name;
					$list .= $name . $temp_sep;
				}
			}

			return $list;
		}

		/**
		 * Check if the Elementor is updated.
		 *
		 * @return boolean if Elementor updated.
		 * @since 1.16.1
		 *
		 */
		static public function is_elementor_updated() {
			if (class_exists('Elementor\Icons_Manager')) {
				return true;
			} else {
				return false;
			}
		}

		/**
		 * Return the new icon name.
		 *
		 * @param string $control_name name of the control.
		 * @return string of the updated control name.
		 * @since 1.16.1
		 *
		 */
		static public function get_new_icon_name($control_name)	{
			if (class_exists('Elementor\Icons_Manager')) {
				return 'new_' . $control_name . '[value]';
			} else {
				return $control_name;
			}
		}

		public static function get_grid_metro_size() {
			return [
				'1:1'   => esc_html__( 'Width 1 - Height 1', 'autohive-core' ),
				'1:2'   => esc_html__( 'Width 1 - Height 2', 'autohive-core' ),
				'1:0.7' => esc_html__( 'Width 1 - Height 70%', 'autohive-core' ),
				'1:1.3' => esc_html__( 'Width 1 - Height 130%', 'autohive-core' ),
				'2:1'   => esc_html__( 'Width 2 - Height 1', 'autohive-core' ),
				'2:2'   => esc_html__( 'Width 2 - Height 2', 'autohive-core' ),
			];
		}

		public static function paging_nav( $query = false ) {
			global $wp_query, $wp_rewrite;

			if ( $query === false ) {
				$query = $wp_query;
			}

			// Don't print empty markup if there's only one page.
			if ( $query->max_num_pages < 2 ) {
				return;
			}

			if ( get_query_var( 'paged' ) ) {
				$paged = get_query_var( 'paged' );
			} elseif ( get_query_var( 'page' ) ) {
				$paged = get_query_var( 'page' );
			} else {
				$paged = 1;
			}

			$page_num_link = html_entity_decode( get_pagenum_link() );
			$query_args    = array();
			$url_parts     = explode( '?', $page_num_link );

			if ( isset( $url_parts[1] ) ) {
				wp_parse_str( $url_parts[1], $query_args );
			}

			$page_num_link = esc_url( remove_query_arg( array_keys( $query_args ), $page_num_link ) );
			$page_num_link = trailingslashit( $page_num_link ) . '%_%';

			$format = '';
			if ( $wp_rewrite->using_index_permalinks() && ! strpos( $page_num_link, 'index.php' ) ) {
				$format = 'index.php/';
			}
			if ( $wp_rewrite->using_permalinks() ) {
				$format .= user_trailingslashit( $wp_rewrite->pagination_base . '/%#%', 'paged' );
			} else {
				$format .= '?paged=%#%';
			}

			// Set up paginated links.

			$args  = array(
				'base'      => $page_num_link,
				'format'    => $format,
				'total'     => $query->max_num_pages,
				'current'   => max( 1, $paged ),
				'mid_size'  => 1,
				'add_args'  => array_map( 'urlencode', $query_args ),
				'prev_text' => '<span class="fas fa-chevron-left">',
				'next_text' => '<span class="fas fa-chevron-right"></span>',
				'type'      => 'array',
			);
			$pages = paginate_links( $args );

			if ( is_array( $pages ) ) {
				echo '<ul class="page-pagination">';
				foreach ( $pages as $page ) {
					printf( '<li>%s</li>', $page );
				}
				echo '</ul>';
			}
		}

		/**
		 * Enqueue svg to the media.
		 * @return void
		 */
		function autohive_svg_mime_types($mimes) {
			$mimes['svg'] = 'image/svg+xml';
			return $mimes;
		}


		public static function image_placeholder( $width, $height ) {
			echo '<img src="http://via.placeholder.com/' . $width . 'x' . $height . '?text=' . esc_attr__( 'No+Image', 'autohive-core' ) . '" alt="' . esc_attr__( 'Thumbnail', 'autohive-core' ) . '"/>';
		}

		public static function autohive_hex_to_rgb($hex, $alpha = false) {
			$hex      = str_replace('#', '', $hex);
			$length   = strlen($hex);
			$rgb['r'] = hexdec($length == 6 ? substr($hex, 0, 2) : ($length == 3 ? str_repeat(substr($hex, 0, 1), 2) : 0));
			$rgb['g'] = hexdec($length == 6 ? substr($hex, 2, 2) : ($length == 3 ? str_repeat(substr($hex, 1, 1), 2) : 0));
			$rgb['b'] = hexdec($length == 6 ? substr($hex, 4, 2) : ($length == 3 ? str_repeat(substr($hex, 2, 1), 2) : 0));
			if ( $alpha ) {
				$rgb['a'] = $alpha;
			}
			return $rgb;
		}

		/**
		 * Day links to archive page
		 */
		public static function the_day_link() {
			$archive_year   = get_the_time( 'Y' );
			$archive_month  = get_the_time( 'm' );
			$archive_day    = get_the_time( 'd' );
			echo get_day_link( $archive_year, $archive_month, $archive_day);
		}


		/**
		 * Check if the url is external or nofollow
		 * @param $settings_key
		 * @param bool $is_echo
		 * @return string|void
		 */
		public static function the_button( $settings_key, $is_echo = true ) {
			if ( $is_echo == true ) {
				echo !empty($settings_key['url']) ? "href='{$settings_key['url']}'" : '';
				echo $settings_key['is_external'] == true ? 'target="_blank"' : '';
				echo $settings_key['nofollow'] == true ? 'rel="nofollow"' : '';
				if ( !empty($settings_key['custom_attributes']) ) {
					$attrs = explode(',', $settings_key['custom_attributes']);
					if(is_array($attrs)){
						foreach($attrs as $data) {
							$data_attrs = explode('|', $data);
							echo esc_attr( $data_attrs[0].'='.$data_attrs[1] );
						}
					}
				}
			}
		}


		/**
		 * Category array
		 * @param string $term
		 * @return array
		 */
		public static function get_the_category_array( $term = 'category' ) {
			$cats = get_terms( array(
				'taxonomy' => $term,
				'hide_empty' => true
			));
			
			$cat_array = [];
			if ( is_array( $cats ) ) {
				foreach ($cats as $cat) {
					$cat_array[$cat->slug] = $cat->name;
				}
			}
			return $cat_array;
		}


		/**
		 * Get the first category name
		 * @param string $term
		 * @return string
		 */
		public static function get_the_first_category( $term = 'category' ) {
			$cats = get_the_terms(get_the_ID(), $term);
			$cat  = is_array($cats) ? $cats[0]->name : '';
			return esc_html($cat);
		}

		/**
		 * Get the first category link
		 * @param string $term
		 * @return string
		 */
		public static function get_the_first_category_link( $term = 'category' ) {
			$cats = get_the_terms(get_the_ID(), $term);
			$cat  = is_array($cats) ? get_category_link($cats[0]->term_id) : '';
			return esc_url($cat);
		}

		/**
		 * Get post excerpt length
		 * @param $settings
		 * @param $settings_key
		 * @param int $default
		 * @return string
		 */
		public static function get_the_excerpt_length ( $settings, $settings_key, $default = 10 ) {
			$excerpt_length = !empty($settings[$settings_key]) ? $settings[$settings_key] : $default;
			$excerpt = get_the_excerpt() ? wp_trim_words(get_the_excerpt(), $excerpt_length, '...') : wp_trim_words(get_the_content(), $excerpt_length, '...');
			return $excerpt;
		}

		/**
		 * Get title excerpt length
		 * @param $settings
		 * @param $settings_key
		 * @param int $default
		 * @return string|void
		 */
		public static function get_the_title_length ( $settings, $settings_key, $default = 10 ) {
			$title_length = !empty($settings[$settings_key]) ? $settings[$settings_key] : $default;
			$title = get_the_title() ? wp_trim_words(get_the_title(), $title_length, '') : the_title();
			return $title;
		}

		/**
		 * Get Reading Time
		 * @return string
		 */
		public static function get_the_reading_time() {
			$content = get_post_field( 'post_content', get_the_ID() );
			$word_count = str_word_count( strip_tags( $content ) );
			$reading_time = ceil($word_count / 200);

			if ($reading_time == 1) {
				$timer = esc_html__( " minute read", 'saasland-core' );
			} else {
				$timer = esc_html__( " minutes read", 'saasland-core' );
			}

			$total_reading_time = $reading_time . $timer;

			return esc_html($total_reading_time);
		}

		/**
		 * Get the limit latter
		 * @param $string
		 * @param $limit_length
		 * @param string $suffix
		 */
		public function get_the_limit_latter($string, $limit_length, $suffix = '...' ) {

			if (strlen($string) > $limit_length) {
				return strip_shortcodes(substr($string, 0, $limit_length) . $suffix);
			}
			else {
				return strip_shortcodes(esc_html($string));
			}
		}

		/**
		 * @param $post_author_id
		 * @param int $size
		 * @param string $args
		 * @return false|mixed|void
		 */
		public static function get_the_author_avatar( $post_author_id, $size = '', $args = '' ) {
			$post_author_id = get_the_author_meta( 'user_email' );
			$post_author = get_avatar( $post_author_id, $size, '', '', $args );
			return $post_author;
		}

		/**
		 * @param $html_data
		 * @return mixed
		 */
		public function html_return($html_data) {
			return $html_data;
		}


		/**
		 * @param string  $content Text content to filter.
		 * @return string Filtered content containing only the allowed HTML.
		 */
		public static function kses_post($content) {
			$allowed_tag = array(
				'strong' => [],
				'br' => [],
				'p' => [
					'class' => [],
					'style' => [],
				],
				'i' => [
					'class' => [],
					'style' => [],
				],
				'ul' => [
					'class' => [],
					'style' => [],
				],
				'li' => [
					'class' => [],
					'style' => [],
				],
				'span' => [
					'class' => [],
					'style' => [],
				],
				'a' => [
					'href' => [],
					'class' => [],
					'title' => []
				],
				'div' => [
					'class' => [],
					'style' => [],
				],
				'h1' => [
					'class' => [],
					'style' => []
				],
				'h2' => [
					'class' => [],
					'style' => []
				],
				'h3' => [
					'class' => [],
					'style' => []
				],
				'h4' => [
					'class' => [],
					'style' => []
				],
				'h5' => [
					'class' => [],
					'style' => []
				],
				'h6' => [
					'class' => [],
					'style' => []
				],
				'img' => [
					'class' => [],
					'style' => [],
					'height' => [],
					'width' => [],
					'src' => [],
					'srcset' => [],
					'alt' => [],
				],

			);
			return wp_kses($content, $allowed_tag);
		}

		/**
		 * Event Tab data
		 * @param $getCats
		 * @param $schedule_cats
		 * @return array
		 */
		public function get_tab_data( $getCats, $schedule_cats ) {
			$y = [];
			foreach ( $getCats as $val ) {

				$t = [];
				foreach( $schedule_cats as $data ) {
					if( $data['tab_title'] == $val ) {
						$t[] = $data;
					}
				}
				$y[$val] = $t;
			}
			return $y;
		}


		public function get_percentage_sale_price ($original_price = 100, $sale_price = 50) {
			$old_price = $original_price;
			$current_price = $sale_price;
			$percent  = ($old_price - $current_price ) * 100 / $old_price;
			return $percent;
		}

		public static function autohive_car_meta( $brand ='', $pickup_location ='')
		{

			$args = array(
				'post_type'     => 'autohive_rent',
				'posts_per_page' => -1,
				'orderby'       => 'meta_value',
			);

			if (!empty($brand)) {
				$args['tax_query'] = [
					['taxonomy' => 'autohive_rent_brand', 'field'   => 'slug', 'terms' => $brand]
				];
			}

			if (!empty($pickup_location)) {
				$args['meta_query'] = [
					['key' => 'pickup_location', 'value'   => $pickup_location]
				];
			}


			$postslist = get_posts($args);
			$pickupLocation = array();
			foreach ($postslist as $key => $item) {
				$pickupLocation[$key] = get_post_meta($item->ID, 'pickup_location', true);
			}
			return $pickupLocation;
		}

		// Dealer post Count
		public static function autohive_dealer_posts( $dealer_id = '') {
			$args = array(
				'post_type'     => 'autohive_listing',
				'posts_per_page' => -1,
				'orderby'       => 'meta_value',
			);

			if ( !empty( $dealer_id ) ) {
				$args['meta_query'] = [
					['key' => 'select_dealer', 'value'   => $dealer_id]
				];
			}

			$postslist = get_posts($args);
			$dealer_ids = array();
			foreach ($postslist as $key => $item) {
				$dealer_ids[$key] = get_post_meta($item->ID, 'select_dealer', true);
			}
			return $dealer_ids;
		}

	}


	Autohive_Core_Helper::instance();
}
