<?php
namespace Autohive\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Elementor\{Controls_Manager,
	Core\Kits\Documents\Tabs\Global_Colors,
	Group_Control_Background,
	Group_Control_Border,
	Group_Control_Box_Shadow,
	Group_Control_Text_Shadow,
	Icons_Manager,
	Widget_Base,
	Group_Control_Typography,
	Repeater,
	Utils};

class Autohive_Button extends Widget_Base {

	public function get_name() {
		return 'autohive_button';
	}

	public function get_title() {
		return esc_html__( 'Autohive Button', 'autohive-core' );
	}

	public function get_icon() {
		return 'eicon-button';
	}

	public function get_categories() {
		return [ 'autohive-elements' ];
	}

	/**
	 * Name: register_controls
	 * Desc: Register controls for these widgets
	 * Params: no params
	 * Return: @void
	 * Since: @1.0.0
	 * Package: @autohive
	 * Author: Themetags
	 */
	protected function register_controls() {
		$this-> autohive_elementor_content_control();
		$this-> autohive_elementor_style_control();
	}


	/**
	 * Name: autohive_elementor_content_control
	 * Desc: Register content
	 * Params: no params
	 * Return: @void
	 * Since: @1.0.0
	 * Package: @autohive
	 * Author: Themetags
	 */
	public function autohive_elementor_content_control() {
		
		//================= Buttons ======================//
		$this->start_controls_section(
			'section_button', [
				'label' => esc_html__( 'Button', 'autohive-core' ),
			]
		);

		$this->add_control(
			'btn_type', [
				'label' => esc_html__( 'Style', 'autohive-core' ),
				'type' => Controls_Manager::SELECT,
				'default' => '1',
				'options' => [
					'1' => esc_html__( 'Default', 'autohive-core' ),
					'2' => esc_html__( 'Hover Top to Bottom', 'autohive-core' ),
					'3' => esc_html__( 'Hover Left to Right Transition', 'autohive-core' ),
					'4' => esc_html__( 'Dual Button', 'autohive-core' ),
				],
			]
		);

		$this->add_control(
			'btn_label_top', [
				'label' => esc_html__( 'Button Top Text', 'autohive-core' ),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__( 'Get it on', 'autohive-core' ),
                'condition' => [
                    'btn_type' => '4'
                ]
			]
		);

		$this->add_control(
			'text', [
				'label' => esc_html__( 'Text', 'autohive-core' ),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__( 'Click here', 'autohive-core' ),
			]
		);

		$this->add_control(
			'link', [
				'label' => esc_html__( 'Link', 'autohive-core' ),
				'type' => Controls_Manager::URL,
				'default' => [
					'url' => '#',
				],
			]
		);

		$this->add_responsive_control(
			'align', [
				'label' => esc_html__( 'Alignment', 'autohive-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left'    => [
						'title' => esc_html__( 'Left', 'autohive-core' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'autohive-core' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'autohive-core' ),
						'icon' => 'eicon-text-align-right',
					],
					'justify' => [
						'title' => esc_html__( 'Justified', 'autohive-core' ),
						'icon' => 'eicon-text-align-justify',
					],
				],
				'prefix_class' => 'elementor%s-align-',
				'default' => '',
			]
		);

		$this->add_control(
			'size', [
				'label' => esc_html__( 'Size', 'autohive-core' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'sm',
				'options' => [
					'xs' => esc_html__( 'Extra Small', 'autohive-core' ),
					'sm' => esc_html__( 'Small', 'autohive-core' ),
					'md' => esc_html__( 'Medium', 'autohive-core' ),
					'lg' => esc_html__( 'Large', 'autohive-core' ),
					'xl' => esc_html__( 'Extra Large', 'autohive-core' ),
				],
				'style_transfer' => true,
			]
		);

		$this->add_control(
			'selected_icon', [
				'label' => esc_html__( 'Icon', 'autohive-core' ),
				'type' => Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'skin' => 'inline',
				'label_block' => false,
			]
		);

		$this->add_control(
			'icon_align', [
				'label' => esc_html__( 'Icon Position', 'autohive-core' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'left' => esc_html__( 'Before', 'autohive-core' ),
					'right' => esc_html__( 'After', 'autohive-core' ),
				],
				'default' => 'left',
				'condition' => [
					'selected_icon[value]!' => '',
				],
			]
		);

		$this->add_control(
			'icon_indent', [
				'label' => esc_html__( 'Icon Spacing', 'autohive-core' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'max' => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .elementor-button .elementor-align-icon-right' => 'margin-left: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .elementor-button .elementor-align-icon-left' => 'margin-right: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'button_css_id', [
				'label' => esc_html__( 'Button ID', 'autohive-core' ),
				'type' => Controls_Manager::TEXT,
				'dynamic' => [
					'active' => true,
				],
				'default' => '',
				'title' => esc_html__( 'Add your custom id WITHOUT the Pound key. e.g: my-id', 'autohive-core' ),
				'description' => sprintf(
				/* translators: 1: Code open tag, 2: Code close tag. */
					esc_html__( 'Please make sure the ID is unique and not used elsewhere on the page this form is displayed. This field allows %1$sA-z 0-9%2$s & underscore chars without spaces.', 'autohive-core' ),
					'<code>',
					'</code>'
				),
				'separator' => 'before',

			]
		);

		$this->end_controls_section();// End Button

	}


	/**
	 * Name: autohive_elementor_style_control
	 * Desc: Register style content
	 * Params: no params
	 * Return: @void
	 * Since: @1.0.0
	 * Package: @autohive
	 * Author: Themetags
	 */
	public function autohive_elementor_style_control () {

		$this->start_controls_section(
			'style_btn', [
				'label' => esc_html__( 'Button', 'autohive-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(), [
				'name' => 'typography',
				'selector' => '{{WRAPPER}} .elementor-button',
			]
		);

		$this->add_group_control(
			Group_Control_Text_Shadow::get_type(),
			[
				'name' => 'text_shadow',
				'selector' => '{{WRAPPER}} .elementor-button',
			]
		);

		//=== Button Tabs : Normal & Hover
		$this->start_controls_tabs( 'tabs_button_style' );

		// Button Normal
		$this->start_controls_tab(
			'tab_button_normal', [
				'label' => esc_html__( 'Normal', 'autohive-core' ),
			]
		);

		$this->add_control(
			'button_text_color', [
				'label' => esc_html__( 'Text Color', 'autohive-core' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .elementor-button' => 'fill: {{VALUE}}; color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(), [
				'name' => 'background',
				'label' => esc_html__( 'Background', 'autohive-core' ),
				'types' => [ 'classic', 'gradient' ],
				'exclude' => [ 'image' ],
				'selector' => '{{WRAPPER}} .elementor-button',
				'fields_options' => [
					'background' => [
						'default' => 'classic',
					],
				],
			]
		);

		$this->end_controls_tab(); //End Button Normal

		// Button Hover
		$this->start_controls_tab(
			'tab_button_hover', [
				'label' => esc_html__( 'Hover', 'autohive-core' ),
			]
		);

		$this->add_control(
			'hover_color', [
				'label' => esc_html__( 'Text Color', 'autohive-core' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .elementor-button:hover, {{WRAPPER}} .elementor-button:focus' => 'color: {{VALUE}};',
					'{{WRAPPER}} .elementor-button:hover svg, {{WRAPPER}} .elementor-button:focus svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(), [
				'name' => 'button_background_hover',
				'label' => esc_html__( 'Background', 'autohive-core' ),
				'types' => [ 'classic', 'gradient' ],
				'exclude' => [ 'image' ],
				'selector' => '
				    {{WRAPPER}} .elementor-button:hover, 
                    {{WRAPPER}} .elementor-button:focus, 
                    {{WRAPPER}} .autohive-btn-wrapper .hover_top_bottom::before,
                    {{WRAPPER}} .autohive-btn-wrapper .hover_left_to_right::before
                ',
				'fields_options' => [
					'background' => [
						'default' => 'classic',
					],
				],
			]
		);

		$this->add_control(
			'button_hover_border_color', [
				'label' => esc_html__( 'Border Color', 'autohive-core' ),
				'type' => Controls_Manager::COLOR,
				'condition' => [
					'border_border!' => '',
				],
				'selectors' => [
					'{{WRAPPER}} .elementor-button:hover, {{WRAPPER}} .elementor-button:focus' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'hover_animation', [
				'label' => esc_html__( 'Hover Animation', 'autohive-core' ),
				'type' => Controls_Manager::HOVER_ANIMATION,
			]
		);

		$this->end_controls_tab(); // End Button Hover

		$this->end_controls_tabs(); // End Button Tabs

		$this->add_group_control(
			Group_Control_Border::get_type(), [
				'name' => 'border',
				'selector' => '{{WRAPPER}} .elementor-button',
				'separator' => 'before',
			]
		);

		$this->add_control(
			'border_radius', [
				'label' => esc_html__( 'Border Radius', 'autohive-core' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .elementor-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(), [
				'name' => 'button_box_shadow',
				'selector' => '{{WRAPPER}} .elementor-button',
			]
		);

		$this->add_responsive_control(
			'text_padding', [
				'label' => esc_html__( 'Padding', 'autohive-core' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors' => [
					'{{WRAPPER}} .elementor-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'separator' => 'before',
			]
		);

		$this->end_controls_section(); // End Button Style


        //======================= Start Button Icon Style ===========================//
		$this->start_controls_section(
			'style_btn_icon', [
				'label' => esc_html__( 'Icon', 'autohive-core' ),
				'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'btn_type' => '4'
                ]
			]
		);

		$this->add_control(
			'icon_size', [
				'label' => esc_html__( 'Size', 'autohive-core' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => '',
				],
				'selectors' => [
					'{{WRAPPER}} .elementor-button .elementor-button-icon' => 'font-size: {{SIZE}}{{UNIT}};',
				],
			]
		);


		$this->end_controls_section(); //End Button Icon Style

	}

	/**
	 * Name: render
	 * Desc: Widgets Render
	 * Params: no params
	 * Return: @void
	 * Since: @1.0.0
	 * Package: @autohive
	 * Author: Themetags
	 */
	protected function render() {

		$settings     = $this->get_settings_for_display();
		extract( $settings );

        if ( !empty( $settings['btn_type'] == '4' ) ) {
	        $this->add_render_attribute( 'wrapper', 'class', 'elementor-button-wrapper apps-btn-group d-flex align-items-center justify-content-center flex-wrap' );
        } else {
	        $this->add_render_attribute( 'wrapper', 'class', 'elementor-button-wrapper autohive-btn-wrapper' );
        }

		if ( ! empty( $settings['link']['url'] ) ) {
			$this->add_link_attributes( 'button', $settings['link'] );
			$this->add_render_attribute( 'button', 'class', 'elementor-button-link' );
		}

		$this->add_render_attribute( 'button', 'class', 'elementor-button' );
		$this->add_render_attribute( 'button', 'role', 'button' );

		if ( ! empty( $settings['button_css_id'] ) ) {
			$this->add_render_attribute( 'button', 'id', $settings['button_css_id'] );
		}

		if ( ! empty( $settings['size'] ) ) {
			$this->add_render_attribute( 'button', 'class', 'elementor-size-' . $settings['size'] );
		}

		if ( $settings['hover_animation'] ) {
			$this->add_render_attribute( 'button', 'class', 'elementor-animation-' . $settings['hover_animation'] );
		}

        if ( !empty($settings['btn_type'] == '2') ) {
            $this->add_render_attribute('button', 'class', 'hover_top_bottom');
        }

        if ( !empty($settings['btn_type'] == '3') ) {
            $this->add_render_attribute('button', 'class', 'hover_left_to_right');
        }

        if ( !empty($settings['btn_type'] == '4') ) {
	        $this->add_render_attribute('button', 'class', 'app-btn');
        }

        if ( !empty( $settings['btn_type'] == '4' )) {
            ?>
            <div <?php $this->print_render_attribute_string( 'wrapper' ); ?>>
                <a <?php $this->print_render_attribute_string( 'button' ); ?>>
			        <?php $this->render_dual_btn_text(); ?>
                </a>
            </div>
            <?php
        } else {
            ?>
            <div <?php $this->print_render_attribute_string( 'wrapper' ); ?>>
                <a <?php $this->print_render_attribute_string( 'button' ); ?>>
			        <?php $this->render_text(); ?>
                </a>
            </div>
            <?php
        }

	}


	/**
	 * Render button text.
	 *
	 * Render button widget text.
	 *
	 * @since 1.5.0
	 * @access protected
	 */
	protected function render_text() {
		$settings = $this->get_settings_for_display();

		$migrated = isset( $settings['__fa4_migrated']['selected_icon'] );
		$is_new = empty( $settings['icon'] ) && Icons_Manager::is_migration_allowed();

		if ( ! $is_new && empty( $settings['icon_align'] ) ) {
			$settings['icon_align'] = $this->get_settings( 'icon_align' );
		}

		$this->add_render_attribute( [
			'content-wrapper' => [
				'class' => 'elementor-button-content-wrapper',
			],
			'icon-align' => [
				'class' => [
					'elementor-button-icon',
					'elementor-align-icon-' . $settings['icon_align'],
				],
			],
			'text' => [
				'class' => 'elementor-button-text',
			],
		] );

		$this->add_inline_editing_attributes( 'text', 'none' );
		?>
		<span <?php $this->print_render_attribute_string( 'content-wrapper' ); ?>>
			<?php if ( ! empty( $settings['icon'] ) || ! empty( $settings['selected_icon']['value'] ) ) : ?>
				<span <?php $this->print_render_attribute_string( 'icon-align' ); ?>>
                    <?php if ( $is_new || $migrated ) :
                        Icons_Manager::render_icon( $settings['selected_icon'], [ 'aria-hidden' => 'true' ] );
                    else : ?>
                        <i class="<?php echo esc_attr( $settings['icon'] ); ?>" aria-hidden="true"></i>
                    <?php endif; ?>
                </span>
			<?php endif; ?>
			<span <?php $this->print_render_attribute_string( 'text' ); ?>><?php $this->print_unescaped_setting( 'text' ); ?></span>
		</span>
		<?php
	}

    protected function render_dual_btn_text() {
	    $settings = $this->get_settings_for_display();

	    $migrated = isset( $settings['__fa4_migrated']['selected_icon'] );
	    $is_new = empty( $settings['icon'] ) && Icons_Manager::is_migration_allowed();

	    if ( ! $is_new && empty( $settings['icon_align'] ) ) {
		    $settings['icon_align'] = $this->get_settings( 'icon_align' );
	    }

	    $this->add_render_attribute( [
		    'content-wrapper' => [
			    'class' => 'elementor-button-content-wrapper align-items-center',
		    ],
		    'icon-align' => [
			    'class' => [
				    'elementor-button-icon',
				    'elementor-align-icon-' . $settings['icon_align'],
			    ],
		    ],
		    'text' => [
			    'class' => 'elementor-button-text app-btn-content text-start',
		    ],
	    ] );

	    $this->add_inline_editing_attributes( 'text', 'none' );

        ?>
        <span <?php $this->print_render_attribute_string( 'content-wrapper' ); ?>>
			<?php if ( ! empty( $settings['icon'] ) || ! empty( $settings['selected_icon']['value'] ) ) : ?>
                <span <?php $this->print_render_attribute_string( 'icon-align' ); ?>>
                    <?php if ( $is_new || $migrated ) :
                        Icons_Manager::render_icon( $settings['selected_icon'], [ 'aria-hidden' => 'true' ] );
                    else : ?>
                        <i class="<?php echo esc_attr( $settings['icon'] ); ?>" aria-hidden="true"></i>
                    <?php endif; ?>
                </span>
			<?php endif; ?>

            <div <?php $this->print_render_attribute_string( 'text' ); ?>>
                <span><?php echo esc_html($settings['btn_label_top']) ?></span>
                <strong><?php $this->print_unescaped_setting( 'text' ); ?></strong>
            </div>
		</span>
        <?php

    }

}