<?php
/**
 * Plugin Name: Autohive Core
 * Plugin URI: https://themeforest.net/user/themetags/portfolio
 * Description: This plugin adds the core features to the Autohive WordPress theme. You must have to install this plugin to get all the features included with the theme.
 * Version: 1.0.0
 * Author: ThemeTags
 * Author URI: https://themeforest.net/user/themetags/portfolio
 * Text domain: autohive-core
 */

if (!defined('ABSPATH'))
	die('-1');

/**
 * Currently plugin version.
 * Start at version 2.0.0 and use SemVer - https://semver.org
 * Rename this for your plugin and update it as you release new versions.
 */
define('AUTOHIVE_CORE_VERSION', '1.0.0');

/**
 * Constant for the plugins
 */
define('AUTOHIVE_CORE_DIR', plugin_dir_path( __FILE__ ));
define('AUTOHIVE_PLUGIN_URL', plugins_url() . '/autohive-core');
define('AUTOHIVE_PLUGIN_ASSETS_URL', plugins_url() . '/autohive-core/assets');

define('AUTOHIVE_CORE_DIR_CSS', AUTOHIVE_PLUGIN_ASSETS_URL .'/css');
define('AUTOHIVE_CORE_DIR_JS', AUTOHIVE_PLUGIN_ASSETS_URL .'/js');
define('AUTOHIVE_CORE_DIR_VEND', AUTOHIVE_PLUGIN_ASSETS_URL .'/vendors');


// Make sure the same class is not loaded twice in free/premium versions.
if (!class_exists('Autohive_Core')) {
	/**
	 * Main Autohive Core Class
	 *
	 * The main class that initiates and runs the Autohive Core plugin.
	 *
	 * @since 1.0.0
	 */
	final class Autohive_Core {
		/**
		 * Autohive Core Version
		 *
		 * Holds the version of the plugin.
		 *
		 * @since 1.0.0
		 * 
		 * @var string The plugin version.
		 */
		const VERSION = '1.0';

		/**
		 * Minimum Elementor Version
		 *
		 * Holds the minimum Elementor version required to run the plugin.
		 *
		 * @since 1.0.0
		 * 
		 * @var string Minimum Elementor version required to run the plugin.
		 */
		const MINIMUM_ELEMENTOR_VERSION = '3.3.0';

		/**
		 * Minimum PHP Version
		 *
		 * Holds the minimum PHP version required to run the plugin.
		 *
		 * @since 2.0.0
		 * 
		 * @var string Minimum PHP version required to run the plugin.
		 */
		const  MINIMUM_PHP_VERSION = '5.6';

		/**
		 * Instance
		 *
		 * Holds a single instance of the `Autohive_Core` class.
		 *
		 * @since 1.0.0
		 *
		 * @access private
		 * @static
		 *
		 * @var Autohive_Core A single instance of the class.
		 */
		private static $_instance = null;

		/**
		 * The loader that's responsible for maintaining and registering all hooks that power
		 * the plugin.
		 *
		 * @since    1.0.0
		 * @access   protected
		 * @var      Autohive_Core    $loader    Maintains and registers all hooks for the plugin.
		 */
		protected $loader;

		/**
		 * Instance
		 *
		 * Ensures only one instance of the class is loaded or can be loaded.
		 *
		 * @return Autohive_Core An instance of the class.
		 * @since 1.0.0
		 *
		 * @access public
		 * @static
		 *
		 */
		public static function instance() {
			if (is_null(self::$_instance)) {
				self::$_instance = new self();
			}

			return self::$_instance;
		}

		/**
		 * Clone
		 *
		 * Disable class cloning.
		 *
		 * @return void
		 * @since 1.0.0
		 *
		 * @access protected
		 *
		 */
		public function __clone() {
			// Cloning instances of the class is forbidden
			_doing_it_wrong(__FUNCTION__, esc_html__('Cheatin&#8217; huh?', 'autohive-core'), '2.0.0');
		}

		/**
		 * Wakeup
		 *
		 * Disable unserializing the class.
		 *
		 * @return void
		 * @since 1.0.0
		 *
		 * @access protected
		 *
		 */
		public function __wakeup() {
			// Unserializing instances of the class is forbidden.
			_doing_it_wrong(__FUNCTION__, esc_html__('Cheatin&#8217; huh?', 'autohive-core'), '2.0.0');
		}

		/**
		 * Constructor
		 *
		 * Initialize the Autohive Core plugins.
		 *
		 * @since 1.0.0
		 *
		 * @access public
		 */
		public function __construct() {
			$this->core_includes();
			$this->init_hooks();
			do_action('autohive_core_loaded');
		}

		/**
		 * Include Files
		 *
		 * Load core files required to run the plugin.
		 *
		 * @since 1.0.0
		 *
		 * @access public
		 */
		public function core_includes()	{
			// Extra
			require_once __DIR__ . '/inc/mailchimp.php';
			require_once __DIR__ . '/inc/extra.php';

            // Register Custom Post Type
			require_once __DIR__ . '/inc/post-type/autohive-listing.php';
			require_once __DIR__ . '/inc/post-type/autohive-listing-rent.php';
			require_once __DIR__ . '/inc/post-type/electric-car.php';
			require_once __DIR__ . '/inc/post-type/listing-dealer.php';
			require_once __DIR__ . '/inc/post-type/Listing_motor.php';
			require_once __DIR__ . '/inc/post-type/mega-menu.php';
			require_once __DIR__ . '/inc/post-type/footer.php';
			require_once __DIR__ . '/inc/post-type/faq.php';
			require_once __DIR__ . '/inc/post-type/service.php';


			//Elementor Widgets
			require_once __DIR__ . '/elementor/elementor_init.php';

            // Elementor Classes helper functions
			require_once __DIR__ . '/inc/classes/class-autohive-helper.php';
            require_once __DIR__ . '/inc/classes/class-autohive-woo-helper.php';
			require_once __DIR__ . '/inc/classes/class-advance-search.php';
			require_once __DIR__ . '/inc/classes/Advance_Search_Autoparts.php';

            // Elementor Classes script's
			require_once __DIR__ . '/inc/classes/class-enqueue-scripts.php';

			// Sidebar Widgets
			require_once __DIR__ . '/wp-widgets/widgets.php';
		}

		/**
		 * Init Hooks
		 *
		 * Hook into actions and filters.
		 *
		 * @since 1.0.0
		 *
		 * @access private
		 */
		private function init_hooks() {
			add_action('init', [$this, 'i18n']);
			add_action('plugins_loaded', [$this, 'init']);
			add_action( 'init', [ $this, 'autohive_gutenberg_block_init' ] );
		}

		/**
		 * Load Textdomain
		 *
		 * Load plugin localization files.
		 *
		 * @since 1.0.0
		 *
		 * @access public
		 */
		public function i18n() {
			load_plugin_textdomain('autohive-core', false, plugin_basename(dirname(__FILE__)) . '/languages');
		}

		/**
		 * Init Autohive Core
		 *
		 * Load the plugin after Elementor (and other plugins) are loaded.
		 *
		 * @since 1.0.0
		 * @since 2.0.0 The logic moved from a standalone function to this class method.
		 *
		 * @access public
		 */
		public function init() {

			if (!did_action('elementor/loaded')) {
				add_action('admin_notices', [$this, 'admin_notice_missing_main_plugin']);
				return;
			}

			// Check for required Elementor version
			if (!version_compare(ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=')) {
				add_action('admin_notices', [$this, 'admin_notice_minimum_elementor_version']);
				return;
			}

			// Check for required PHP version
			if (version_compare(PHP_VERSION, self::MINIMUM_PHP_VERSION, '<')) {
				add_action('admin_notices', [$this, 'admin_notice_minimum_php_version']);
				return;
			}

		}

		/**
		 * Admin notice
		 *
		 * Warning when the site doesn't have Elementor installed or activated.
		 *
		 * @since 1.1.0
		 * @since 2.0.0 Moved from a standalone function to a class method.
		 *
		 * @access public
		 */
		public function admin_notice_missing_main_plugin() {
			$message = sprintf(
			/* translators: 1: Autohive Core: Elementor */
				esc_html__('"%1$s" requires "%2$s" to be installed and activated.', 'autohive-core'),
				'<strong>' . esc_html__('Autohive Core', 'autohive-core') . '</strong>',
				'<strong>' . esc_html__('Elementor', 'autohive-core') . '</strong>'
			);
			printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
		}

		/**
		 * Admin notice
		 *
		 * Warning when the site doesn't have a minimum required Elementor version.
		 *
		 * @since 1.1.0
		 * @since 2.0.0 Moved from a standalone function to a class method.
		 *
		 * @access public
		 */
		public function admin_notice_minimum_elementor_version() {
			$message = sprintf(
			/* translators: 1: Autohive Core: Elementor 3: Required Elementor version */
				esc_html__('"%1$s" requires "%2$s" version %3$s or greater.', 'autohive-core'),
				'<strong>' . esc_html__('Autohive Core', 'autohive-core') . '</strong>',
				'<strong>' . esc_html__('Elementor', 'autohive-core') . '</strong>',
				self::MINIMUM_ELEMENTOR_VERSION
			);
			printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
		}

		/**
		 * Admin notice
		 *
		 * Warning when the site doesn't have a minimum required PHP version.
		 *
		 * @since 2.0.0
		 *
		 * @access public
		 */
		public function admin_notice_minimum_php_version() {
			$message = sprintf(
			/* translators: 1: Autohive Core 2: PHP 3: Required PHP version */
				esc_html__('"%1$s" requires "%2$s" version %3$s or greater.', 'autohive-core'),
				'<strong>' . esc_html__('Autohive Elements', 'autohive-core') . '</strong>',
				'<strong>' . esc_html__('PHP', 'autohive-core') . '</strong>',
				self::MINIMUM_PHP_VERSION
			);
			printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
		}

		/*
		 * Gutenberg Block Register
		 */
		public function autohive_gutenberg_block_init() {
			register_block_type(
				'autohive-block/newsletter-widget',
				[
					'render_callback' => [ $this, 'render_newsletter' ],
					'attributes'      => [
						'btn_text'       => [
							'default' => __('Subscribe', 'autohive-core'),
							'type'    => 'string',
						],
						'content'       => [
							'default' => __('Subscribe to our newsletter to receive early discount offers.', 'autohive-core'),
							'type'    => 'string',
						],
					],
				]
			);
		}

		public function render_newsletter( $attributes ) {
			ob_start(); ?>
			<div class="wp-widget-block-newsletter">
				<p class="description"><?php echo $attributes['content'] ?></p>

				<form action="<?php echo admin_url( 'admin-ajax.php' ); ?>" method="post" class="footer-newsletter-form widget-newsletter" data-autohive-form="newsletter-subscribe">
					<input type="hidden" name="action" value="autohive_mailchimp_subscribe">

					<div class="newsletter-inner d-flex">
						<input type="email" name="email" class="form-control" id="newsletter-form-email" placeholder="<?php echo esc_attr__( 'Email', 'autohive-core' ); ?>" required>
						<button type="submit" name="submit" id="newsletter-submit" class="newsletter-submit autohive-btn">
							<?php echo esc_html($attributes['btn_text']); ?>
							<i class="fa fa-circle-o-notch fa-spin"></i>
						</button>
					</div>

					<div class="form-result alert">
						<div class="content"></div>
					</div><!-- /.form-result-->
				</form><!-- /.newsletter-form -->
				</div>
			<!-- /.wp-widget-block-newsletter -->
			<?php
			return ob_get_clean();
		}
	}
}
// Make sure the same function is not loaded twice in free/premium versions.

if (!function_exists('autohive_core_load')) {
	/**
	 * Load Autohive Core
	 *
	 * Main instance of Autohive_Core.
	 *
	 * @since 1.0.0
	 * @since 2.0.0 The logic moved from this function to a class method.
	 */
	function autohive_core_load()	{
		return Autohive_Core::instance();
	}

	// Run Autohive Core
	autohive_core_load();
}